<?php
require_once 'config.php';

// Check if user is logged in
if (!isset($_SESSION['user_name'])) {
    header("Location: index.php");
    exit();
}

// Menu groups
$menu_groups = array(
    'Administration',
    'Accounts',
    'Dag',
    'Data Capture',
    'Inquiries',
    'Inventory',
    'Master Files',
    'Reports',
    'Reports-Customer',
    'Services',
    'Stores',
    'System Utilities'
);

// Store permissions in memory (session)
// if (!isset($_SESSION['user_permissions'])) {
$_SESSION['user_permissions'] = array();

// Get all permissions for the user
foreach ($menu_groups as $group) {
    $sql = "SELECT docname, link FROM view_userpermission WHERE username=? AND grp=?";
    if ($stmt = mysqli_prepare($conn, $sql)) {
        mysqli_stmt_bind_param($stmt, "ss", $_SESSION['user_name'], $group);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_store_result($stmt);
        mysqli_stmt_bind_result($stmt, $docname, $link);

        while (mysqli_stmt_fetch($stmt)) {
            // Store both docname (display) and link (file path)
            $_SESSION['user_permissions'][$group][] = array(
                'docname' => $docname,
                'url' => $link
            );
        }
        mysqli_stmt_close($stmt);
    }
}
// }

// Get selected group
$selected_group = isset($_GET['group']) ? $_GET['group'] : (isset($menu_groups[0]) ? $menu_groups[0] : '');
?>

<!DOCTYPE html>
<html>

<head>
    <title>ERP System - Menu</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
        }

        .header {
            background:
                <?php echo $colors['primary']; ?>
            ;
            color: white;
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }

        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .logout-btn {
            background:
                <?php echo $colors['danger']; ?>
            ;
            color: white;
            border: none;
            padding: 8px 15px;
            border-radius: 5px;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s;
        }

        .logout-btn:hover {
            background: #c0392b;
            transform: translateY(-2px);
        }

        .tabs-container {
            background:
                <?php echo $colors['secondary']; ?>
            ;
            display: flex;
            flex-wrap: wrap;
            gap: 2px;
            padding: 10px 20px;
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .tab {
            background:
                <?php echo $colors['info']; ?>
            ;
            color: white;
            padding: 12px 25px;
            border: none;
            cursor: pointer;
            font-weight: 600;
            font-size: 14px;
            transition: all 0.3s;
            border-radius: 5px 5px 0 0;
        }

        .tab:hover {
            background: #2980b9;
            transform: translateY(-2px);
        }

        .tab.active {
            background: white;
            color:
                <?php echo $colors['primary']; ?>
            ;
            box-shadow: 0 -2px 10px rgba(0, 0, 0, 0.1);
        }

        .menu-container {
            padding: 30px;
            max-width: 1400px;
            margin: 0 auto;
        }

        .menu-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .menu-item {
            background: linear-gradient(45deg, #ffffff, #f8f9fa);
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
            font-weight: 600;
            font-size: 16px;
            color:
                <?php echo $colors['dark']; ?>
            ;
            min-height: 80px;
            display: flex;
            align-items: center;
            justify-content: center;
            word-break: break-word;
            position: relative;
            overflow: hidden;
        }

        .menu-item:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
            border-color:
                <?php echo $colors['info']; ?>
            ;
        }

        .menu-item:focus {
            outline: none;
            border-color:
                <?php echo $colors['focus']; ?>
            ;
            box-shadow: 0 0 0 3px rgba(230, 126, 34, 0.3);
            transform: scale(1.02);
        }

        .menu-item.empty {
            background: linear-gradient(45deg, #f8f9fa, #e9ecef);
            color: #6c757d;
            cursor: not-allowed;
            font-style: italic;
            opacity: 0.7;
        }

        .welcome-message {
            background: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
            border-left: 5px solid
                <?php echo $colors['success']; ?>
            ;
        }

        .welcome-message h2 {
            color:
                <?php echo $colors['primary']; ?>
            ;
            margin-bottom: 5px;
        }

        .company-name {
            color:
                <?php echo $colors['info']; ?>
            ;
            font-weight: 600;
        }

        .color-indicator {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 5px;
            background: linear-gradient(90deg, #3498db, #2ecc71);
        }

        .shortcut-key {
            position: absolute;
            top: 5px;
            right: 5px;
            background:
                <?php echo $colors['warning']; ?>
            ;
            color: white;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            font-size: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
        }
    </style>
</head>

<body>
    <div class="header">
        <h1>ERP SYSTEM MENU</h1>
        <div class="user-info">
            <span>Welcome, <strong><?php echo $_SESSION['user_name']; ?></strong></span>
            <span class="company-name">[<?php echo $_SESSION['company']; ?>]</span>
            <button class="logout-btn" onclick="logout()">Logout (Ctrl+Q)</button>
        </div>
    </div>

    <div class="tabs-container">
        <?php foreach ($menu_groups as $group): ?>
            <button class="tab <?php echo $selected_group == $group ? 'active' : ''; ?>"
                onclick="selectTab('<?php echo urlencode($group); ?>')" data-group="<?php echo $group; ?>" tabindex="0">
                <?php echo $group; ?>
            </button>
        <?php endforeach; ?>
    </div>

    <div class="menu-container">

        <div class="menu-grid">
            <?php
            if ($selected_group && isset($_SESSION['user_permissions'][$selected_group])) {
                $items = $_SESSION['user_permissions'][$selected_group];
                if (!empty($items)) {
                    $counter = 1;
                    foreach ($items as $item):
                        // Backward compatibility check if session is old
                        $display_name = is_array($item) ? $item['docname'] : $item;
                        $url = is_array($item) ? $item['url'] : $item;
                        ?>
                        <div class="menu-item" tabindex="0" onclick="selectMenu('<?php echo $url; ?>')"
                            onkeypress="handleKeyPress(event, '<?php echo $url; ?>')" data-shortcut="<?php echo $counter; ?>"
                            data-url="<?php echo $url; ?>">
                            <div class="color-indicator"></div>
                            <?php echo htmlspecialchars($display_name); ?>
                        </div>
                        <?php
                        $counter++;
                    endforeach;
                } else {
                    echo '<div class="menu-item empty">No permissions for this group</div>';
                }
            } else {
                echo '<div class="menu-item empty">Select a menu group</div>';
            }
            ?>
        </div>
    </div>

    <script>
        // Store menu items in memory for keyboard navigation
        var menuItems = [];
        var currentTabIndex = 0;
        var currentMenuItemIndex = 0;

        function selectTab(group) {
            window.location.href = 'menu.php?group=' + group;
        }

        function selectMenu(docname) {
            var url = docname;
            if (!url.endsWith('.php')) {
                url += '.php';
            }
            window.open(url, '_blank');
        }

        function logout() {
            window.location.href = 'logout.php';
        }

        // Keyboard Navigation
        document.addEventListener('keydown', function (e) {
            // Logout shortcut
            if (e.ctrlKey && e.key === 'q') {
                logout();
                e.preventDefault();
            }

            // Tab navigation between menu groups
            if (e.key === 'Tab' && !e.shiftKey) {
                if (document.activeElement.classList.contains('tab')) {
                    // If focus is on tabs, move to menu items
                    e.preventDefault();
                    var firstMenuItem = document.querySelector('.menu-item[tabindex]');
                    if (firstMenuItem) firstMenuItem.focus();
                }
            }

            // Alt + Number shortcuts for menu items
            if (e.altKey && e.key >= '1' && e.key <= '9') {
                var index = parseInt(e.key) - 1;
                var items = document.querySelectorAll('.menu-item[data-shortcut]');
                if (items[index]) {
                    selectMenu(items[index].dataset.url);
                }
                e.preventDefault();
            }
        });

        // Focus style management
        document.addEventListener('focusin', function (e) {
            if (e.target.classList.contains('menu-item')) {
                e.target.style.borderColor = '<?php echo $colors["focus"]; ?>';
                e.target.style.boxShadow = '0 0 0 3px rgba(230, 126, 34, 0.3)';
            }
        });

        document.addEventListener('focusout', function (e) {
            if (e.target.classList.contains('menu-item')) {
                e.target.style.borderColor = '#e9ecef';
                e.target.style.boxShadow = 'none';
            }
        });

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function () {
            // Focus first tab if none selected
            var activeTab = document.querySelector('.tab.active');
            if (activeTab) {
                activeTab.focus();
            }

            // Store all menu items for keyboard navigation
            menuItems = Array.from(document.querySelectorAll('.menu-item[tabindex]'));
        });

        function handleKeyPress(event, docname) {
            if (event.key === 'Enter' || event.key === ' ') {
                selectMenu(docname);
                event.preventDefault();
            }
        }
    </script>
</body>

</html>