(function() {
    'use strict';

    let currentRefno = '';
    let masterItems = [];
    let capturedBarcodes = [];

    const els = {
        purchaseNo: document.getElementById('bc-purchase-no'),
        supplier: document.getElementById('bc-supplier'),
        btnFind: document.getElementById('bc-btn-find'),
        btnNew: document.getElementById('bc-btn-new'),
        btnSave: document.getElementById('bc-btn-save'),
        btnClear: document.getElementById('bc-btn-clear'),
        btnLoad: document.getElementById('bc-btn-load'),
        barcode: document.getElementById('bc-barcode'),
        line: document.getElementById('bc-line'),
        masterTable: document.getElementById('bc-master-table').getElementsByTagName('tbody')[0],
        captureTable: document.getElementById('bc-capture-table').getElementsByTagName('tbody')[0],
        status: document.getElementById('bc-status')
    };

    // Status message helper
    function showStatus(message, type) {
        if (!els.status) return;
        els.status.textContent = message;
        els.status.className = 'label label-' + (type || 'info');
        els.status.style.display = 'inline-block';
        
        setTimeout(() => {
            els.status.style.display = 'none';
        }, 5000);
    }

    // Clear all data
    function clearAll() {
        currentRefno = '';
        masterItems = [];
        capturedBarcodes = [];
        
        if (els.purchaseNo) els.purchaseNo.value = '';
        if (els.supplier) els.supplier.value = '';
        if (els.barcode) {
            els.barcode.value = '';
            els.barcode.disabled = true;
        }
        if (els.line) {
            els.line.innerHTML = '<option value="">Select item...</option>';
            els.line.disabled = true;
        }
        
        renderMasterTable();
        renderCaptureTable();
        
        if (els.btnLoad) els.btnLoad.disabled = true;
        if (els.btnSave) els.btnSave.disabled = true;
        
        showStatus('Form cleared', 'info');
    }

    // Render master items table
    function renderMasterTable() {
        if (!els.masterTable) return;
        
        if (masterItems.length === 0) {
            els.masterTable.innerHTML = '<tr><td colspan="5" class="text-center text-muted"><em>Enter Purchase No and click Find to load items</em></td></tr>';
            return;
        }
        
        els.masterTable.innerHTML = masterItems.map(item => `
            <tr>
                <td>${escapeHtml(item.STK_NO || '')}</td>
                <td>${escapeHtml(item.DESCRIPT || '')}</td>
                <td>${escapeHtml(item.MANUFACTURE || '')}</td>
                <td>${escapeHtml(item.EXDATE || '')}</td>
                <td class="text-right">${escapeHtml(item.QTY || '0')}</td>
            </tr>
        `).join('');
    }

    // Render capture table
    function renderCaptureTable() {
        if (!els.captureTable) return;
        
        if (capturedBarcodes.length === 0) {
            els.captureTable.innerHTML = '<tr><td colspan="8" class="text-center text-muted"><em>No barcodes captured yet</em></td></tr>';
            return;
        }
        
        els.captureTable.innerHTML = capturedBarcodes.map((item, idx) => `
            <tr>
                <td class="text-right">${idx + 1}</td>
                <td>${escapeHtml(item.stkNo)}</td>
                <td>${escapeHtml(item.description)}</td>
                <td>${escapeHtml(item.manufacture)}</td>
                <td>${escapeHtml(item.exdate)}</td>
                <td class="text-right">${escapeHtml(item.qty)}</td>
                <td>${escapeHtml(item.barcode)}</td>
                <td class="text-center">
                    <button type="button" class="btn btn-xs btn-danger" onclick="barcodeCenterApp.removeBarcode(${idx})">
                        <i class="fa fa-trash"></i>
                    </button>
                </td>
            </tr>
        `).join('');
    }

    // Populate line picker
    function populateLinePicker() {
        if (!els.line) return;
        
        const options = masterItems.map(item => 
            `<option value="${escapeHtml(item.STK_NO)}">${escapeHtml(item.STK_NO)} - ${escapeHtml(item.DESCRIPT)}</option>`
        ).join('');
        
        els.line.innerHTML = '<option value="">Select item...</option>' + options;
        els.line.disabled = false;
    }

    // Find purchase
    function findPurchase() {
        const refno = els.purchaseNo ? els.purchaseNo.value.trim() : '';
        
        // If no refno entered, show popup list
        if (!refno) {
            showPurchaseListPopup();
            return;
        }

        showStatus('Searching...', 'info');
        
        fetch('barcode_center_data.php?Command=find_purchase&refno=' + encodeURIComponent(refno))
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    loadPurchaseData(refno, data);
                } else {
                    alert(data.message || 'Purchase not found');
                    showStatus(data.message || 'Not found', 'danger');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error loading purchase data');
                showStatus('Error loading data', 'danger');
            });
    }

    // Show purchase list popup
    function showPurchaseListPopup() {
        $('#purchaseListModal').modal('show');
        
        const tableBody = document.querySelector('#purchase-list-table tbody');
        if (tableBody) {
            tableBody.innerHTML = '<tr><td colspan="5" class="text-center"><i class="fa fa-spinner fa-spin"></i> Loading...</td></tr>';
        }
        
        fetch('barcode_center_data.php?Command=list_all_purchases')
            .then(response => response.json())
            .then(data => {
                if (data.success && data.purchases) {
                    renderPurchaseList(data.purchases);
                } else {
                    if (tableBody) {
                        tableBody.innerHTML = '<tr><td colspan="5" class="text-center text-danger">Error loading purchases</td></tr>';
                    }
                }
            })
            .catch(error => {
                console.error('Error:', error);
                if (tableBody) {
                    tableBody.innerHTML = '<tr><td colspan="5" class="text-center text-danger">Error loading purchases</td></tr>';
                }
            });
    }

    // Render purchase list in modal
    function renderPurchaseList(purchases) {
        const tableBody = document.querySelector('#purchase-list-table tbody');
        if (!tableBody) return;
        
        if (purchases.length === 0) {
            tableBody.innerHTML = '<tr><td colspan="5" class="text-center text-muted">No purchases found</td></tr>';
            return;
        }
        
        tableBody.innerHTML = purchases.map(p => `
            <tr>
                <td>${escapeHtml(p.REFNO || '')}</td>
                <td>${escapeHtml(p.SDATE || '')}</td>
                <td>${escapeHtml(p.SUP_NAME || '')}</td>
                <td>${escapeHtml(p.LCNO || '')}</td>
                <td class="text-center">
                    <button type="button" class="btn btn-xs btn-primary" onclick="barcodeCenterApp.selectPurchase('${escapeHtml(p.REFNO)}')">
                        <i class="fa fa-check"></i> Select
                    </button>
                </td>
            </tr>
        `).join('');
    }

    // Select purchase from popup
    function selectPurchase(refno) {
        $('#purchaseListModal').modal('hide');
        
        if (els.purchaseNo) {
            els.purchaseNo.value = refno;
        }
        
        showStatus('Loading purchase...', 'info');
        
        fetch('barcode_center_data.php?Command=find_purchase&refno=' + encodeURIComponent(refno))
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    loadPurchaseData(refno, data);
                } else {
                    alert(data.message || 'Purchase not found');
                    showStatus(data.message || 'Not found', 'danger');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error loading purchase data');
                showStatus('Error loading data', 'danger');
            });
    }

    // Load purchase data (extracted from findPurchase)
    function loadPurchaseData(refno, data) {
        currentRefno = refno;
        masterItems = data.items || [];
        capturedBarcodes = [];
        
        if (els.supplier) els.supplier.value = data.purchase.SUP_NAME || '';
        
        renderMasterTable();
        renderCaptureTable();
        populateLinePicker();
        
        if (els.barcode) els.barcode.disabled = false;
        if (els.btnLoad) els.btnLoad.disabled = false;
        if (els.btnSave) els.btnSave.disabled = false;
        
        showStatus(`Found ${masterItems.length} items`, 'success');
        
        if (els.barcode) els.barcode.focus();
    }

    // Add barcode
    function addBarcode() {
        const barcodeVal = els.barcode ? els.barcode.value.trim() : '';
        const stkNo = els.line ? els.line.value : '';
        
        if (!barcodeVal) {
            alert('Please enter or scan a barcode');
            if (els.barcode) els.barcode.focus();
            return;
        }
        
        if (!stkNo) {
            alert('Please select an item to link to');
            if (els.line) els.line.focus();
            return;
        }
        
        // Check if barcode already exists
        if (capturedBarcodes.some(b => b.barcode === barcodeVal)) {
            alert('This barcode has already been captured');
            if (els.barcode) {
                els.barcode.value = '';
                els.barcode.focus();
            }
            return;
        }
        
        const item = masterItems.find(i => i.STK_NO === stkNo);
        if (!item) {
            alert('Selected item not found');
            return;
        }
        
        capturedBarcodes.push({
            stkNo: item.STK_NO,
            description: item.DESCRIPT,
            manufacture: item.MANUFACTURE,
            exdate: item.EXDATE,
            qty: 1,
            barcode: barcodeVal
        });
        
        renderCaptureTable();
        
        if (els.barcode) {
            els.barcode.value = '';
            els.barcode.focus();
        }
        
        showStatus('Barcode added', 'success');
    }

    // Remove barcode
    function removeBarcode(index) {
        if (confirm('Remove this barcode?')) {
            capturedBarcodes.splice(index, 1);
            renderCaptureTable();
            showStatus('Barcode removed', 'info');
        }
    }

    // Save barcodes
    function saveBarcodes() {
        if (capturedBarcodes.length === 0) {
            alert('No barcodes to save');
            return;
        }
        
        if (!confirm(`Save ${capturedBarcodes.length} barcode(s) for Purchase No: ${currentRefno}?`)) {
            return;
        }
        
        showStatus('Saving...', 'info');
        
        const formData = new FormData();
        formData.append('refno', currentRefno);
        formData.append('barcodes', JSON.stringify(capturedBarcodes));
        
        fetch('barcode_center_data.php?Command=save_barcodes', {
            method: 'POST',
            body: formData
        })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Barcodes saved successfully!');
                    showStatus('Saved successfully', 'success');
                    clearAll();
                } else {
                    alert('Error saving: ' + (data.message || 'Unknown error'));
                    showStatus('Save failed', 'danger');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error saving barcodes');
                showStatus('Save error', 'danger');
            });
    }

    // Helper function to escape HTML
    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return String(text).replace(/[&<>"']/g, m => map[m]);
    }

    // Event listeners
    function initEventListeners() {
        if (els.btnFind) {
            els.btnFind.addEventListener('click', findPurchase);
        }
        
        if (els.purchaseNo) {
            els.purchaseNo.addEventListener('keyup', e => {
                if (e.key === 'Enter') findPurchase();
            });
        }
        
        if (els.btnLoad) {
            els.btnLoad.addEventListener('click', addBarcode);
        }
        
        if (els.barcode) {
            els.barcode.addEventListener('keyup', e => {
                if (e.key === 'Enter') addBarcode();
            });
        }
        
        if (els.btnSave) {
            els.btnSave.addEventListener('click', saveBarcodes);
        }
        
        if (els.btnClear) {
            els.btnClear.addEventListener('click', clearAll);
        }
        
        if (els.btnNew) {
            els.btnNew.addEventListener('click', clearAll);
        }
    }

    // Initialize
    function init() {
        initEventListeners();
        clearAll();
        showStatus('Ready', 'info');
    }

    // Public API
    window.barcodeCenterApp = {
        removeBarcode: removeBarcode,
        selectPurchase: selectPurchase
    };

    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

})();
